/**
 *
 *  Copyright © 2016 Collabora Ltd.
 *
 *  SPDX-License-Identifier: MPL-2.0
 *  This Source Code Form is subject to the terms of the Mozilla Public
 *  License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  Copyright (c) 2017 EDAG Engineering GmbH
 *	Edited by:
 *
 *  @file CAmRoutingSenderGstreamer.cpp
 *  @author Johannes Hartmann, Johannes.Hartmann@edag.de
 *  @date 13 March 2017
 *  @brief File containing the implementation of the Gstreamer Routing Plugin.
 *
 *  The Gstreamer Routing Plugin uses Gstreamer for connecting sources and sinks
 *  (e.g. Bluetooth, Alsa) and lets audio data flowing. An Alsa monitoring is set up
 *  to get information about the sample rate of the usb-alsa device. In the case of
 *  a changing the sample rate the gstreamer pipeline will be reconfigured. This is
 *  done in a callback function.
 *
 */

 #include <string.h>
 #include <math.h>
 #include <sys/stat.h>
 #include <fcntl.h>
 #include <thread>
 /* Audio Manager Utilties headers */
 #include <AudioManagerUtilities/CAmLogWrapper.h>
 #include <cm/audiomanager/customtypes.h>
 /* Gstreamer Routing Adapter headers */
 #include "CAmRoutingSenderGstreamer.h"
 #include "spi_tclMySPINSndCtrlMonitor.h"

using namespace am;
extern am::IAmLogContext &routingGstreamer;

namespace am {

static GstBusSyncReply bus_handler(GstBus *bus, GstMessage *message, CAmRoutingSenderGstreamer *thiz);

/**
 * @brief Callback funktion for the Alsa monitoring. Used to reconfigure the gstreamer pipeline
 *        in the case of a changing the sample rate.
 * @param[in] obj CAmRoutingSenderGstreamer object
 * @param[in] 32SampleRate Sample rate of the alsa device
 * @param[in] bEndPointenabled
 * @return void
 */
extern "C" void sampleRateChanged_CB(void* obj, uint32_t u32SampleRate, bool bEndPointEnabled)
{
	auto *data = (std::tuple<CAmRoutingSenderGstreamer *,  am_sourceID_t>*) obj;
//
	CAmRoutingSenderGstreamer* gstRaObject = std::get< 0 >(*data);
	am_sourceID_t sourceID = std::get< 1 >(*data);

	routingGstreamer.info(__FUNCTION__, ":sampleRateChanged_CB called. New SampleRate=",
				u32SampleRate, ", bEndPointEnabled=", bEndPointEnabled, " for sourceId:", sourceID);

	gstRaObject->reconfigurePipeline(sourceID, u32SampleRate, bEndPointEnabled);
}

static uint8_t pipeDebugCounter = 0;
/**
 * @brief Contains the disconnection sequence
 * @param[in] *pad Pad of the source_bin
 * @param[in] *pipeline Handle of the Gstreamer pipeline
 * @return void
 */
static void disconnect_source(GstPad *pad, GstElement *pipeline)
{
    routingGstreamer.info(__FUNCTION__, ":disconnect_source called");

    GstPad *mixersink = NULL;
    GstElement *mixer = NULL;
    GstElement *source = NULL;
    GstElement *sink = NULL;
    GstPad *sinkghost = NULL;

    sinkghost = gst_pad_get_peer(pad);
    if(sinkghost)
    {
    	gst_pad_unlink(pad, sinkghost);

		mixersink = gst_ghost_pad_get_target((GstGhostPad *) sinkghost);
		if(mixersink)
		{
			mixer = gst_pad_get_parent_element(mixersink);
			gst_element_release_request_pad(mixer, mixersink);
		}
		else
		{
			routingGstreamer.error(__FUNCTION__, ":Could not get ghostpad target");
		}

		source = gst_pad_get_parent_element(pad);
		if(source)
		{
			routingGstreamer.info(__FUNCTION__, ":Remove a source from pipeline ");
			gst_object_ref(GST_OBJECT(source));
			if(!gst_bin_remove(GST_BIN(pipeline), source))
			{
				routingGstreamer.error(__FUNCTION__, ":Remove a source from pipeline failed");
			}
		}
		else
		{
			routingGstreamer.error(__FUNCTION__, ":Could not get ghostpad target");
		}

		sink = gst_pad_get_parent_element(sinkghost);
		if(sink)
		{
			/* Remove ghostpad */
			gst_element_remove_pad(sink, sinkghost);
			if(sink->numsinkpads == 0)
			{
				/* remove sink from pipeline */
				/* change state of pipeline if no source is connected to the sink and remove sink from pipeline */
				routingGstreamer.info(__FUNCTION__, ":Change state of pipeline -- GST_STATE_NULL ");
				if(gst_element_set_state(pipeline, GST_STATE_NULL) == GST_STATE_CHANGE_FAILURE)
				{
					routingGstreamer.error(__FUNCTION__, ":Change element state failed");
				}
				routingGstreamer.info(__FUNCTION__, ":Remove a sink from pipeline ");
				gst_object_ref(GST_OBJECT(sink));
				if(!gst_bin_remove(GST_BIN(pipeline), sink))
				{
					routingGstreamer.error(__FUNCTION__, ":Remove a sink from pipeline failed");
				}
			}
		}
		else
		{
			routingGstreamer.error(__FUNCTION__, ":Could not get sink.");
		}
    }
    else
    {
    	routingGstreamer.error(__FUNCTION__, ":Could not get sinkghost.");
    }

	gst_object_unref(mixersink);
	gst_object_unref(mixer);
    gst_object_unref(source);
	gst_object_unref(sink);
    gst_object_unref(sinkghost);

    gst_object_unref(pad);
}

/**
 * @brief Data structure for pad probe callback function
 */
typedef struct
{
    am_connectionID_t connectionID;     /**< Connection ID */
    CAmRoutingSenderGstreamer *thiz;    /**< Pointer to object */
    GstElement *pipeline;               /**< Pointer to the Gstreamer pipeline */
} ProbeData;

/**
 * @brief Callback function of pad blocked
 * @param[in] *pad The pad which is blocked.
 * @param[in] *probe_data Structure which contains the handle, connectionID, pipeline and pointer to object
 * @return void
 */
static void callback_pipeline_ramped_down(GstPad * pad, gboolean blocked, ProbeData *probe_data)
{
    (void) blocked;

	routingGstreamer.info(__FUNCTION__, ":callback_pipeline_ramped_down called");

    disconnect_source(pad, probe_data->pipeline);

    routingGstreamer.info(__FUNCTION__, ":asynchronously disconnected connection ID ", probe_data->connectionID);
#ifdef USE_GSTREAMER_VERSION_1_0

#else
    gst_pad_set_blocked(pad, false);
#endif
}
#if 0
static void callback_sourcePadBlocked(GstPad *pad, gboolean blocked, gpointer user_data)
{
    (void)pad;
    (void)user_data;
    routingGstreamer.info(__FUNCTION__, "callback_sourcePadBlocked with blocked=", blocked);
}
#endif

static gboolean callback_pad_event_function(GstPad *pad, GstMiniObject *mini_obj, std::tuple<CAmRoutingSenderGstreamer *,  am_sourceID_t> *data)
{
	CAmRoutingSenderGstreamer* gstRaObject = std::get< 0 >(*data);
	am_sourceID_t sourceID = std::get< 1 >(*data);

	GstEvent *event = (GstEvent*) mini_obj;

	routingGstreamer.info(__FUNCTION__, ": pad:",(std::string)GST_PAD_NAME(pad), " id:", pad, ": sourceId:",sourceID);

	switch (GST_EVENT_TYPE (event))
	{
		case GST_EVENT_EOS:
		{
			routingGstreamer.info(__FUNCTION__, " EOS event");
			gstRaObject->sendEOSNotificationToAM(sourceID);
		}break;
		default:
		{
			routingGstreamer.info(__FUNCTION__, " other event (not handled)", (std::string)gst_event_type_get_name(GST_EVENT_TYPE (event)));
		}
	}

	return true;
}

//static std::mutex am_GStRA_padCBLock;
static void callback_source_pad_added(GstElement* element, GstPad* new_pad, gpointer* nextElementInPipe)
{
    //am_GStRA_padCBLock.lock();

    bool alreadyCorrectConnected = false;
    routingGstreamer.info(__FUNCTION__);
    if (nextElementInPipe == NULL)
    {
        routingGstreamer.error("=> No nextElementInPipe provided to connect the new pad.");
        return;
    }

    GstPad* sinkPad = gst_element_get_static_pad((GstElement*)nextElementInPipe, "sink");

    if(NULL == sinkPad)
    {
        routingGstreamer.error("=> nextElementInPipe do not have a sinkPad.");
        return;
    }

    routingGstreamer.info(__FUNCTION__, ": new pad:",(std::string)GST_PAD_NAME(new_pad), " id:", new_pad);
    routingGstreamer.info(__FUNCTION__, ": from element:", (std::string)GST_ELEMENT_NAME(element), " id:", element);
    routingGstreamer.info(__FUNCTION__, ": connect to:",(std::string)GST_ELEMENT_NAME(nextElementInPipe), " id=",
            nextElementInPipe,"with snkPad:",(std::string)GST_PAD_NAME(sinkPad), "id:", sinkPad);

    if (gst_pad_is_linked(sinkPad) == true)
    {
        /* check if the sinkPad of the given element is already connected to the new_pad */

        GstPad* currentPeerPad = gst_pad_get_peer(sinkPad);

        routingGstreamer.info(__FUNCTION__, " snk already linked with peer=", (std::string)GST_OBJECT_NAME(currentPeerPad), " id=",currentPeerPad);

        if(new_pad == currentPeerPad)
        {
            routingGstreamer.info(__FUNCTION__, "New pad is already correct connected.");
            alreadyCorrectConnected = true;
        }
        else
        {
            if(GST_PAD_LINK_FAILED(gst_pad_unlink(currentPeerPad, sinkPad)))
                routingGstreamer.info(__FUNCTION__, "failed to unlink sink");
        }
        gst_object_unref(currentPeerPad);
    }
    else
        routingGstreamer.info(__FUNCTION__, " snk is not linked");


    if(!alreadyCorrectConnected)
    {
        GstCaps *new_pad_caps = NULL;
        GstStructure *new_pad_struct = NULL;

        //ToDo use for GStreamer 1.0: new_pad_caps = gst_pad_get_current_caps (new_pad);
        new_pad_caps = GST_PAD_CAPS(new_pad);
        new_pad_struct = gst_caps_get_structure (new_pad_caps, 0);

        uint8_t numberOfCaps = gst_caps_get_size(new_pad_caps) ;
        routingGstreamer.info(__FUNCTION__, " there are ", numberOfCaps, " caps structs available");

        const gchar *new_pad_type = NULL;
        new_pad_type = gst_structure_get_name (new_pad_struct);

        int width=0, depth=0, channels=0, rate=0;
        if(!gst_structure_get_int (new_pad_struct, "width", &width) | !gst_structure_get_int (new_pad_struct, "depth", &depth) |
                !gst_structure_get_int (new_pad_struct, "channels", &channels) | !gst_structure_get_int (new_pad_struct, "rate", &rate))
            routingGstreamer.error(__FUNCTION__, " failed to get attribute ");

        routingGstreamer.info(__FUNCTION__, " new_pad_type ", new_pad_type);
        routingGstreamer.info(__FUNCTION__, " width ", width);
        routingGstreamer.info(__FUNCTION__, " depth ", depth);
        routingGstreamer.info(__FUNCTION__, " channels ", channels);
        routingGstreamer.info(__FUNCTION__, " rate ", rate);

        if(!gst_structure_has_name(new_pad_struct, "audio/x-raw-int"))
        {
            const gchar *new_pad_type = NULL;
            new_pad_type = gst_structure_get_name (new_pad_struct);
            routingGstreamer.error(__FUNCTION__, " new Pad format", new_pad_type, " does not fit expectation 'audio/x-raw-int'");
        }

        GstPadLinkReturn success = gst_pad_link(new_pad, sinkPad);

        if (GST_PAD_LINK_SUCCESSFUL(success))
        {
            routingGstreamer.info(__FUNCTION__, " Successfully linked pads");
            if(gst_element_sync_state_with_parent(element))
                routingGstreamer.info(__FUNCTION__, " Successfully synced state with parent");
        }
        else
        {
            routingGstreamer.info(__FUNCTION__, " Failed to link pads, code =", success);
        }

        /*
         * ToDo move the disconnect signal to the disconnect source/sink part
         * Maybe generic approach in the controller to link the elements manually and register signals
         * instead of parsing the string of a part-pipe.
         * */
//        int number = g_signal_handlers_disconnect_by_data(element, nextElementInPipe);
//        routingGstreamer.info("Disconnected ", number, " signal handler");
        gst_object_unref(new_pad_caps);
    }

    gst_object_unref(sinkPad);

    if(!g_signal_handlers_disconnect_by_data(element,nextElementInPipe))
    	routingGstreamer.error(__FUNCTION__, " g_signal_handlers_disconnect_by_data failed. No matches found.");

    std::string nameOfDebgPipe = "pipe";
    nameOfDebgPipe.append(std::to_string(pipeDebugCounter));
    GST_DEBUG_BIN_TO_DOT_FILE((GstBin*)gst_element_get_parent(gst_element_get_parent(element)), GST_DEBUG_GRAPH_SHOW_ALL, nameOfDebgPipe.c_str());

    pipeDebugCounter++;
    //am_GStRA_padCBLock.unlock();
}

/**
 * @brief This function is called when something happens on the pipeline-bus
 * @param[in] *bus Pointer to the bus
 * @param[in] *message Pointer to the bus message
 * @param[in] *thiz Pointer to class object
 * @return Return value is always GST_BUS_PASS
 */
static GstBusSyncReply bus_handler(GstBus *bus, GstMessage *message, CAmRoutingSenderGstreamer *thiz)
{
    (void) bus;
    switch(GST_MESSAGE_TYPE(message))
    {
        case GST_MESSAGE_STATE_CHANGED:
            GstState oldstate, newstate, pending;
            gst_message_parse_state_changed(message, &oldstate, &newstate, &pending);
            routingGstreamer.info(__FUNCTION__, ":source: ", (std::string)GST_ELEMENT_NAME(message->src), ", \told state: ", oldstate, ", \tnew state: ", newstate);
            thiz->dispatchStateChange(message->src, newstate);
            break;
        case GST_MESSAGE_ASYNC_DONE:
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_ASYNC_DONE: ", GST_MESSAGE_TYPE(message));
            break;
        case GST_MESSAGE_STREAM_STATUS:
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_STREAM_STATUS: ", GST_MESSAGE_TYPE(message));

            GstStreamStatusType type;
            GstElement *owner;
            gst_message_parse_stream_status (message, &type, &owner);
            /*
             * Type:
             * GST_STREAM_STATUS_TYPE_CREATE   a new thread need to be created.
             * GST_STREAM_STATUS_TYPE_ENTER    a thread entered its loop function
             * GST_STREAM_STATUS_TYPE_LEAVE    a thread left its loop function
             * GST_STREAM_STATUS_TYPE_DESTROY  a thread is destroyed
             * GST_STREAM_STATUS_TYPE_START    a thread is started
             * GST_STREAM_STATUS_TYPE_PAUSE    a thread is paused
             * GST_STREAM_STATUS_TYPE_STOP     a thread is stopped
             */
            routingGstreamer.info(__FUNCTION__, ":source: ", (std::string)GST_ELEMENT_NAME(message->src), ", type: ", type, ", owner: ", (std::string)GST_ELEMENT_NAME(owner));

            break;
        case GST_MESSAGE_PROGRESS:
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_PROGRESS: ", GST_MESSAGE_TYPE(message));
            break;
        case GST_MESSAGE_ERROR: {

            GError* error = NULL;
            gchar* errorString = NULL;

            gst_message_parse_error(message, &error, &errorString);
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_ERROR: from element", (std::string)(GST_OBJECT_NAME (message->src)), error->message);
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_ERROR: ", errorString ? std::string(errorString) : "n.a.");

            g_error_free(error);
            g_free(errorString);
            break;
        }
        case GST_MESSAGE_BUFFERING:
            gint percent;
            gst_message_parse_buffering (message, &percent);
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_BUFFERING Buffer level in % ", percent);
            break;
        case GST_MESSAGE_EOS:
        {
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_EOS ", GST_MESSAGE_TYPE(message));
            thiz->dispatchStateChangeEOS(message->src, GST_STATE_NULL);
        }break;
        case GST_MESSAGE_CLOCK_LOST:
            routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_CLOCK_LOST ", GST_MESSAGE_TYPE(message));
            break;
        case GST_MESSAGE_ELEMENT:
        {
        	routingGstreamer.warn(__FUNCTION__, ":bus_handler GST_MESSAGE_ELEMENT ", GST_MESSAGE_TYPE(message));
        }break;
        default:
            routingGstreamer.warn(__FUNCTION__, ":bus_handler default: ", GST_MESSAGE_TYPE(message));
            break;
    }
    gst_message_unref (message);

    return GST_BUS_PASS;
}

/** @brief The constructor of class CAmRoutingSenderGstreamer
 *        Initialization of Gstreamer domain
 */
CAmRoutingSenderGstreamer::CAmRoutingSenderGstreamer(am::GStreamer::IAmAsyncRoutingReceive *asynRoutingReceive,
													 am::GStreamer::IAmCustomRoutingSend *routingSender)
	: mpIAmRoutingReceive()
	, mpCAmRoutingReceiverShadow(asynRoutingReceive)
	, m_routingSender(routingSender)
	, threadChangeVolume(NULL)
	, threadChangeSourceVolume(NULL)
	, threadChangeBTSourceState(NULL)
	, mpspi_tclMySPINSndCtrlMonitor()
	, m_gstController(nullptr)
{
	m_routingSender->registerGstreamerAdapter(this);
}

/**
 * @brief The destructor of class CAmRoutingSenderGstreamer
 */
CAmRoutingSenderGstreamer::~CAmRoutingSenderGstreamer()
{
	delete mpCAmRoutingReceiverShadow;
    delete mpspi_tclMySPINSndCtrlMonitor;
    if(this->threadChangeVolume)
    {
        this->threadChangeVolume->join();
        delete this->threadChangeVolume;
        this->threadChangeVolume=NULL;
    }
    if(this->threadChangeSourceVolume)
	{
		this->threadChangeSourceVolume->join();
		delete this->threadChangeSourceVolume;
		this->threadChangeSourceVolume=NULL;
	}
    if(this->threadChangeBTSourceState)
    {
    	this->threadChangeBTSourceState->join();
        delete this->threadChangeBTSourceState;
        this->threadChangeBTSourceState=NULL;
    }
    delete m_gstController;
}

/**
 * @brief Print out a *.dot file with the given gstreamer pipe
 * @param[in] *connection connection to print
 * @return void
 */
void CAmRoutingSenderGstreamer::debugPrintGStreamerPipeToFile( RoutingSenderGstConnection *connection)
{
    std::string nameOfDebgPipe = "pipe";
    nameOfDebgPipe.append(std::to_string(pipeDebugCounter));
    GST_DEBUG_BIN_TO_DOT_FILE((GstBin*)connection->pipeline, GST_DEBUG_GRAPH_SHOW_ALL, nameOfDebgPipe.c_str());

    pipeDebugCounter++;
}

/**
 * @brief The startup interface.
 * @param[in] *pIAmRoutingReceive pointer interface class
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::startupInterface(IAmRoutingReceive* pIAmRoutingReceive)
{
    routingGstreamer.info(__FUNCTION__, ":startupInterface called");
    am_Error_e result = E_UNKNOWN;
    CAmSocketHandler *handler = nullptr;

    if(NULL != pIAmRoutingReceive)
    {
		mpIAmRoutingReceive = pIAmRoutingReceive;
		result = mpIAmRoutingReceive->getSocketHandler(handler);
		if(E_OK == result && (NULL != handler) )
		{
			mpCAmRoutingReceiverShadow->startupInterface(mpIAmRoutingReceive, handler);
			mpspi_tclMySPINSndCtrlMonitor = new spi_tclMySPINSndCtrlMonitor();
			m_gstController = new GStreamer::Controller(mpIAmRoutingReceive, mpCAmRoutingReceiverShadow,
			                                            (GstBusSyncHandler) bus_handler, (void*) this);
		}
		else
		{
			routingGstreamer.error(__FUNCTION__, ":startupInterface getSocketHandler failed");
		}
    }
    else
	{
		routingGstreamer.error(__FUNCTION__, ":startupInterface pIAmRoutingReceive == NULL");
	}
    return result;
}

/**
 * @brief Returns the interface version
 * @param[out] version The interface version
 * @return void
 */
void CAmRoutingSenderGstreamer::getInterfaceVersion(std::string & version) const
{
    routingGstreamer.info(__FUNCTION__, ":getInterfaceVersion called");
    version = RoutingVersion;
}

/**
 * @brief Send an EOS Notification to a given sourceID
 * @param[in] pad where the EOS occurs
 * @return void
 */
void CAmRoutingSenderGstreamer::sendEOSNotificationToAM(am_sourceID_t sourceID)
{
	routingGstreamer.info(__FUNCTION__);

	am_NotificationPayload_s notificationPayload = {
	.type = NT_SOURCE_EOS,
	.value = 1
	};

	this->mpCAmRoutingReceiverShadow->hookSourceNotificationDataChange(sourceID, notificationPayload);
}

/**
 * @brief Create a new pipe from conID
 * @param[IN] conId for the pipeline
 * @param[out] pipeline
 * @param[out] bus for the pipeline
 * @return E_OK
 */
am_Error_e CAmRoutingSenderGstreamer::setNewPipe(const am_connectionID_t connectionID, RoutingSenderGstConnection *connection)
{
	routingGstreamer.info(__FUNCTION__, ":setNewPipe for conID: ",connectionID);

    if (NULL == connection)
    {
        routingGstreamer.error(__FUNCTION__, ":given GstConnection pointer is Null");
        return E_NOT_POSSIBLE;
    }

    std::string pipename = "GStreamer-domain_" + std::to_string((int) connectionID);
    connection->pipeline = gst_pipeline_new(pipename.c_str());
    connection->bus = gst_pipeline_get_bus(GST_PIPELINE(connection->pipeline));

#ifdef USE_GSTREAMER_VERSION_1_0
	gst_bus_set_sync_handler(connection->bus, (GstBusSyncHandler) bus_handler, this, NULL);
#else
	gst_bus_set_sync_handler(connection->bus, (GstBusSyncHandler) bus_handler, this);
#endif
    routingGstreamer.info(__FUNCTION__, ":setNewPipe : ",connection->pipeline);
    routingGstreamer.info(__FUNCTION__, ":setNewBus : ",connection->bus);

	return E_OK;
}

am_Error_e CAmRoutingSenderGstreamer::fillCurrentPipeWithElements(const am_connectionID_t connectionID)
{
	GstState currentState, pending;
	am_Error_e retvalue = E_OK;
	GstElement *sourceElement = NULL, *sinkElement = NULL;
    am_sourceID_t sourceID = 0;
    am_sinkID_t sinkID = 0;

	auto connection = m_gstController->getConnectionItem(connectionID);
	if(nullptr != connection)
	{
		sourceID = connection->src->source.sourceID;
		sinkID = connection->sink->sink.sinkID;

		routingGstreamer.info(__FUNCTION__, ": fill pipe ", connection->pipeline);
		routingGstreamer.info(__FUNCTION__, ": source ID", sourceID, "/ sink ID", sinkID);

		/* wait if there is an asynchronously e.g. sample rate change ongoing  */
		if (gst_element_get_state(connection->pipeline, &currentState, &pending, 100 * GST_MSECOND)
				== GST_STATE_CHANGE_FAILURE)
			routingGstreamer.warn(__FUNCTION__, ":FillPipe: Change element state asynchronously failed, continue");

		/* Start Monitoring Thread for Alsa device hw:UAC2Gadget */
		if(std::string::npos != connection->src->flags.find("GST_RA_LIVE_USB"))
		{
			routingGstreamer.info(__FUNCTION__, ":Start Alsa monitoring in a new thread");
			auto usbCallbackData = new std::tuple<CAmRoutingSenderGstreamer *,  am_sourceID_t >(this, sourceID);
			mpspi_tclMySPINSndCtrlMonitor->vInitMonitor( sampleRateChanged_CB, usbCallbackData);
		}

        sourceElement = connection->src->element;
        sinkElement = connection->sink->element;
        routingGstreamer.info(__FUNCTION__, ": connect source", (std::string)GST_ELEMENT_NAME(sourceElement)," with sink ", (std::string)GST_ELEMENT_NAME(sinkElement));
	}
	else
	{
		routingGstreamer.error(__FUNCTION__, ":connection empty  ", connectionID);
		retvalue = E_NON_EXISTENT;
	}

	if(E_OK == retvalue)
	{
	    retvalue = prepareSource(sourceElement, sinkElement, connection->pipeline) ? E_OK : E_NOT_POSSIBLE;
	}

	if(E_OK == retvalue)
	{
	    retvalue = prepareSink(sinkElement, connection->pipeline) ? E_OK : E_NOT_POSSIBLE;
	}

	if(E_OK == retvalue)
	{
	    retvalue = registerSometimesPadCallbacks(sourceElement) ? E_OK : E_NOT_POSSIBLE;
	}

    if(E_OK == retvalue)
    {
        retvalue = setGhostPadForConnection(connection) ? E_OK : E_NOT_POSSIBLE;
    }

    /* sinkghost will only ref the target (sinkpad), not itself. From example unref to sinkpad is enough. */

	if(E_NOT_POSSIBLE == retvalue)
	{
		routingGstreamer.error(__FUNCTION__, ":\"Not possible\" error detected. DeleteAllPipeElement cleanup.");
		deleteAllPipeElements();
	}

	routingGstreamer.info(__FUNCTION__, ":FillPipe finished", retvalue);

	return retvalue;
}

bool CAmRoutingSenderGstreamer::prepareSource(GstElement* sourceElement, GstElement* sinkElement, GstElement* pipeline)
{
    bool success = true;

    GstPad *source_srcPad = gst_element_get_static_pad(sourceElement, "src");
    if(!source_srcPad)
    {
        routingGstreamer.error(__FUNCTION__, ": only static source pads are supported. ");
        return false;
    }

    if(!gst_pad_is_linked(source_srcPad))
    {
        /* Check if source_srcPad is in pipeline */
        if(!gst_object_has_ancestor(GST_OBJECT(sourceElement), GST_OBJECT(pipeline)))
        {
            routingGstreamer.info(__FUNCTION__, ": adding source to pipeline");
            if(!gst_bin_add(GST_BIN(pipeline), sourceElement))
            {
                routingGstreamer.error(__FUNCTION__, ": add a source to pipeline failed");
                success = false;
            }
        }
    }
    else
    {
        /* Check if source is connected with wrong sink */
        GstPad *tmp_sinkghost = gst_pad_get_peer(source_srcPad);
        GstElement *sink = gst_pad_get_parent_element(tmp_sinkghost);

        if(sink == sinkElement)
        {
            routingGstreamer.info(__FUNCTION__, ": source is already connected, sink OK");
        }
        else
        {
            routingGstreamer.error(__FUNCTION__, ": source is already connected, wrong sink NOT OK");
            success = false;
        }

        gst_object_unref(tmp_sinkghost);
        gst_object_unref(sink);
    }
    gst_object_unref(source_srcPad);
    return success;
}

bool CAmRoutingSenderGstreamer::prepareSink(GstElement* sinkElement, GstElement* pipeline)
{
    bool success = true;
    /* Check if sinkElement is in pipeline */
    if(!gst_object_has_ancestor(GST_OBJECT(sinkElement), GST_OBJECT(pipeline)))
    {
        routingGstreamer.info(__FUNCTION__, ": adding sink to pipeline");
        if(!gst_bin_add(GST_BIN(pipeline), sinkElement))
        {
            routingGstreamer.error(__FUNCTION__, ": add a sink to pipeline failed");
            success = false;
        }
    }
    else
    {
        routingGstreamer.info(__FUNCTION__, ": sink is already part of pipeline");
    }

    return success;
}

bool CAmRoutingSenderGstreamer::registerSometimesPadCallbacks(GstElement* sourceElement)
{
    bool success = true;
    GstIterator* iterator = gst_bin_iterate_elements((GstBin*) sourceElement);
    GstElement* element = NULL, *nextElementInPipe = NULL;
    GstIteratorResult isIteratorValid = GST_ITERATOR_OK;

    /* iterating the source pipeline in reverse order */
    isIteratorValid = gst_iterator_next(iterator, (void**) &element);

    while (isIteratorValid == GST_ITERATOR_OK)
    {
        GstPad *childSrcPad = gst_element_get_static_pad(element, "src");

        std::string name = GST_ELEMENT_NAME(element);
        routingGstreamer.info(__FUNCTION__, ": current srcElement:",element, " with name: ", name);

        if (!childSrcPad)
        {
            routingGstreamer.info(__FUNCTION__, ": dynamic source pad detected");
			/* last element is always volume and so previous element is set */
            (void) g_signal_connect(element, "pad-added", G_CALLBACK(callback_source_pad_added),
                    nextElementInPipe);
        }
        else
        {
            gst_object_unref(childSrcPad);
        }

        GstPad *childSinkPad = gst_element_get_static_pad(element, "sink");
        if(!childSinkPad)
            routingGstreamer.warn(__FUNCTION__, ": current element cannot be used as nextElemtInPipe because it do not provide a sink.");
        else
            nextElementInPipe = element;

        isIteratorValid = gst_iterator_next(iterator, (void**) &element);
    }
    gst_iterator_free(iterator);

    return success;
}

bool CAmRoutingSenderGstreamer::setGhostPadForConnection(RoutingSenderGstConnection *connection)
{
    bool success = false;

    /* mixer is always set as the first element of a sink pipe in our configuration*/
    GstElement* mixer = gst_bin_get_by_name(GST_BIN(connection->sink->element), "mixer");
    if(mixer)
    {
        /* request pas is necessary because mixer is a funnel element and sinks are only on request */
#ifdef USE_GSTREAMER_VERSION_1_0
    	connection->sinkPad = gst_element_get_request_pad(mixer, "sink_%u");
#else
    	connection->sinkPad = gst_element_get_request_pad(mixer, "sink%d");
#endif
       if(connection->sinkPad)
       {
           /* NULL will assign a default name for the sinkghost pad */
          GstPad* ghostPad = gst_ghost_pad_new(NULL, connection->sinkPad);
          gst_pad_set_active(ghostPad, TRUE);

          routingGstreamer.info(__FUNCTION__, ": new ghostPad:",(std::string)GST_OBJECT_NAME(ghostPad), " id=", ghostPad);

          if(gst_element_add_pad(connection->sink->element, ghostPad))
          {
              connection->ghostSinkPad = ghostPad;
              success = true;
          }
          else
              routingGstreamer.error(__FUNCTION__, ": could not add pad");
       }
       else
           routingGstreamer.error(__FUNCTION__, ": could not get sinkpad");

       gst_object_unref(mixer);
    }
    else
        routingGstreamer.error(__FUNCTION__, ": could not get mixer");

    return success;
}

bool CAmRoutingSenderGstreamer::linkElements(GstElement* source, GstPad* sinkPad)
{
    bool success = true;

    GstPad* srcPad = gst_element_get_static_pad(source, "src");
    routingGstreamer.info(__FUNCTION__, ": source =", (std::string)GST_ELEMENT_NAME(source));
    if (GST_PAD_LINK_OK != gst_pad_link(srcPad, sinkPad))
    {
        routingGstreamer.error(__FUNCTION__, ":Could not link pads");
        success = false;
    }
    gst_object_unref(srcPad);

    return success;
}

bool CAmRoutingSenderGstreamer::isSrcSinkInConnectionLinked(RoutingSenderGstConnection *connection)
{
    bool linked = false;

    GstPad* srcPad = gst_element_get_static_pad (connection->src->element, "src");

    if (gst_pad_is_linked(connection->ghostSinkPad) == TRUE)
	{
		GstPad* currentPeerPad = gst_pad_get_peer(connection->ghostSinkPad);
		if(currentPeerPad == srcPad)
			linked = true;
	}

    return linked;
}

bool CAmRoutingSenderGstreamer::linkSrcSinkInConnection(RoutingSenderGstConnection *connection)
{
    bool success = false;
    GstPadLinkReturn ret;

    GstPad* srcPad = gst_element_get_static_pad(connection->src->element, "src");

    if(connection->ghostSinkPad && srcPad)
    {
        routingGstreamer.info(__FUNCTION__, ": source =", (std::string)GST_ELEMENT_NAME(connection->src->element), " with ", (std::string)GST_ELEMENT_NAME(connection->ghostSinkPad));
        routingGstreamer.info(__FUNCTION__, ": connection id = ", connection->id);
		routingGstreamer.info(__FUNCTION__, ": srcpad = ", srcPad, ": sinkpad = ", connection->ghostSinkPad);

        ret = gst_pad_link(srcPad, connection->ghostSinkPad);
        routingGstreamer.info(__FUNCTION__, ":gst_pad_link retvalue: %d", ret);
        if (GST_PAD_LINK_OK == ret)
            success = true;
        else
            routingGstreamer.error(__FUNCTION__, ":Could not link pads");

        gst_object_unref(srcPad);
    }
    else
    {
        routingGstreamer.error(__FUNCTION__, ": no ghostPad set or srcPad found");
        routingGstreamer.error(__FUNCTION__, ": srcPad =", srcPad);
        routingGstreamer.error(__FUNCTION__, ": ghostPad =", connection->ghostSinkPad);
    }

    if(success)
        routingGstreamer.info(__FUNCTION__, " success");

    return success;
}

bool CAmRoutingSenderGstreamer::unLinkSrcSinkInConnection(RoutingSenderGstConnection *connection)
{
    bool success = false;

    GstPad* srcPad = gst_element_get_static_pad(connection->src->element, "src");

    routingGstreamer.info(__FUNCTION__, ": connection id = ", connection->id);
    routingGstreamer.info(__FUNCTION__, ": srcpad = ", srcPad, ": sinkpad = ", connection->ghostSinkPad);

    if(connection->ghostSinkPad && srcPad)
    {
        routingGstreamer.info(__FUNCTION__, ": source =", (std::string)GST_ELEMENT_NAME(connection->src->element));
#if 0
#if USE_GSTREAMER_VERSION_1_0
        gst_pad_add_probe (srcPad, GST_PAD_PROBE_TYPE_BLOCK,
            (GstPadProbeCallback) null, connection->pipeline, NULL);
#else
        if(!gst_pad_set_blocked_async(srcPad, true, callback_sourcePadBlocked, nullptr))
            routingGstreamer.error(__FUNCTION__, ":Could not block source pads");
#endif
#endif
        if (gst_pad_unlink(srcPad, connection->ghostSinkPad))
            success = true;
        else
            routingGstreamer.error(__FUNCTION__, ":Could not unlink pads");
    }
    else
    {
        routingGstreamer.error(__FUNCTION__, ": no ghostPad set or srcPad found");
        routingGstreamer.error(__FUNCTION__, ": srcPad =", srcPad);
        routingGstreamer.error(__FUNCTION__, ": ghostPad =", connection->ghostSinkPad);
    }

	/* unreference the source pad */
    gst_object_unref(srcPad);

    if(success)
        routingGstreamer.info(__FUNCTION__, " success");
    else
    	routingGstreamer.info(__FUNCTION__, " failed");

    return success;
}

am_Error_e CAmRoutingSenderGstreamer::clearGhostPad(RoutingSenderGstConnection *connection)
{
    am_Error_e retvalue = E_OK;

    GstPad *mixerPad = NULL;
    GstElement *mixer = NULL;

    mixerPad = gst_ghost_pad_get_target((GstGhostPad *) connection->ghostSinkPad);
    if(mixerPad)
    {
        mixer = gst_pad_get_parent_element(mixerPad);
        gst_element_release_request_pad(mixer, mixerPad);
        gst_object_unref(mixerPad);
    }
    else
    {
        routingGstreamer.error(__FUNCTION__, ":Could not get ghostpad target");
        retvalue = E_NOT_POSSIBLE;
    }

    mixer = gst_pad_get_parent_element(connection->ghostSinkPad);
    if(mixer)
        gst_element_remove_pad(mixer, connection->ghostSinkPad);
    else
    {
        routingGstreamer.error(__FUNCTION__, ":Could not get ghostpad parent");
        retvalue = E_NOT_POSSIBLE;
    }

    gst_object_unref(mixer);

    if(E_OK == retvalue)
        routingGstreamer.info(__FUNCTION__, " success");

    return retvalue;
}

am_Error_e CAmRoutingSenderGstreamer::removeSourceFromPipe(RoutingSenderGstConnection *connection)
{
    am_Error_e retvalue = E_OK;

    gst_object_ref(GST_OBJECT(connection->src->element));
    if(!gst_bin_remove(GST_BIN(connection->pipeline), connection->src->element))
    {
       routingGstreamer.error(__FUNCTION__, ": Could not remove source from pipeline");
       retvalue = E_NOT_POSSIBLE;
    }

    if(E_OK == retvalue)
        routingGstreamer.info(__FUNCTION__, " success");

    return retvalue;
}

am_Error_e CAmRoutingSenderGstreamer::removeSinkFromPipe(RoutingSenderGstConnection *connection)
{
    am_Error_e retvalue = E_OK;

    gst_object_ref(GST_OBJECT(connection->sink->element));
    if(!gst_bin_remove(GST_BIN(connection->pipeline), connection->sink->element))
    {
       routingGstreamer.error(__FUNCTION__, ": Could not remove sink from pipeline");
       retvalue = E_NOT_POSSIBLE;
    }

    if(E_OK == retvalue)
        routingGstreamer.info(__FUNCTION__, " success");

    return retvalue;
}

/**
 * @brief Clear the pipe of elements
 * @param[in] connectionID to get the pipe to clear
 * @return E_OK if all went well, else E_NOT_X
 */
am_Error_e CAmRoutingSenderGstreamer::clearPipeElements(const am_connectionID_t connectionID)
{
	am_Error_e retvalue = E_OK;
	GstPad *srcpad;
    ProbeData *probe_data;
    GstState currentState, state, pending;

	routingGstreamer.info(__FUNCTION__, ":clearPipeElements called: ", connectionID);

	auto connection = m_gstController->getConnectionItem(connectionID);
	if(nullptr != connection)
	{
		/* wait if there is an asynchronously e.g. sample rate change ongoing  */
		if (gst_element_get_state(connection->pipeline, &currentState, &pending, 100 * GST_MSECOND)
				== GST_STATE_CHANGE_FAILURE)
		{
			routingGstreamer.error(__FUNCTION__, ":AsyCon: Change element state asynchronously failed");
			//retvalue = E_NON_EXISTENT;
		}
		routingGstreamer.info(__FUNCTION__, ":clearPipeElements called currentState ", currentState, "pending ", pending);

		gst_element_get_state(connection->src->element, &state, &pending, 0);
		routingGstreamer.info(__FUNCTION__, ":clearPipeElements src state ", state, "pending ", pending);
        srcpad = gst_element_get_static_pad(connection->src->element, "src");
		if(!srcpad)
		{
			routingGstreamer.error(__FUNCTION__, ":Could not get static pad ");
			retvalue = E_NOT_POSSIBLE;
		}

		if(E_OK == retvalue)
		{
			if(state != GST_STATE_NULL)
			{
				probe_data = g_new0(ProbeData, 1);
				probe_data->connectionID = connectionID;
				probe_data->thiz = this;
				probe_data->pipeline = connection->pipeline;

#ifdef USE_GSTREAMER_VERSION_1_0
				gst_pad_add_probe(srcpad, GST_PAD_PROBE_TYPE_BLOCK, (GstPadProbeCallback) callback_pipeline_ramped_down, probe_data, NULL);
#else
				gst_pad_set_blocked_async_full(srcpad, TRUE, (GstPadBlockCallback) callback_pipeline_ramped_down, probe_data, g_free);
#endif
				routingGstreamer.info(__FUNCTION__, ":asynchronously disconnecting connection ID ", connectionID);
			}
			else
			{
				disconnect_source(srcpad, connection->pipeline);
				routingGstreamer.info(__FUNCTION__, ":synchronously disconnected connection ID ", connectionID);
			}
		}
	}
	else
	{
		routingGstreamer.error(__FUNCTION__, ":no connection with ID ", connectionID);
        retvalue = E_NON_EXISTENT;
	}

	routingGstreamer.info(__FUNCTION__, ":clearPipeElements finished.");

	return retvalue;
}

/**
 * @brief unref the pipe and delete elements
 * @param[in] connectionId to get the elements to delete/unref
 * @return E_OK
 */
am_Error_e CAmRoutingSenderGstreamer::deletePipe(const am_connectionID_t connectionID)
{
	auto connection = m_gstController->getConnectionItem(connectionID);

	routingGstreamer.info(__FUNCTION__, ":deletePipe");

	if(nullptr != connection)
	{

		/* Stop Monitoring Thread for Alsa device hw:UAC2Gadget */
		if(std::string::npos != connection->src->flags.find("GST_RA_LIVE_USB"))
		{
		   routingGstreamer.info(__FUNCTION__, ":Stop Alsa monitoring");
		   mpspi_tclMySPINSndCtrlMonitor->vStopMonitor();
		}

#ifdef USE_GSTREAMER_VERSION_1_0
		gst_bus_set_sync_handler(connection->bus, NULL, NULL, NULL);
#else
    	gst_bus_set_sync_handler(connection->bus, NULL, NULL);
#endif

		routingGstreamer.info(__FUNCTION__, ":unrefPipe ", connection->pipeline);
		gst_object_unref(connection->bus);
		gst_object_unref(connection->pipeline);
	}
	else
	{
		routingGstreamer.warn(__FUNCTION__, ":Pipe for connection does not exist. No delete necessary.");
	}

	routingGstreamer.info(__FUNCTION__, ":deletePipe finished");
	return E_OK;
}

/**
 * @brief When there is a critical error, delete all pipes and elements
 * @return void
 */
void CAmRoutingSenderGstreamer::deleteAllPipeElements(void)
{
    std::map<am_connectionID_t, RoutingSenderGstConnection *> m_connections = m_gstController->getAllConnectionItem();

    routingGstreamer.error(__FUNCTION__, ":deleteAllPipe start");
    for (const auto& item : m_connections)
    {
        routingGstreamer.error(__FUNCTION__, ":delete connectionId", item.second->id);
        syncChangeElementState(item.second->pipeline, GST_STATE_NULL);
        clearPipeElements(item.second->id);
        deletePipe(item.second->id);
    }
    routingGstreamer.error(__FUNCTION__, ":deleteAllPipe finished");
}

am_Error_e CAmRoutingSenderGstreamer::openFileDescriptorSource(RoutingSenderGstConnection* connection)
{
    am_Error_e retvalue = E_OK;
    GstElement *bt_source = NULL;
    bt_source = gst_bin_get_by_name(GST_BIN(connection->src->element), "btSource");
    if(bt_source)
    {
        routingGstreamer.info(__FUNCTION__, ":Open file descriptor BT-Source");
        gint fd;
        fd = open (BluetoothSmartphone::getMediaPipe(), O_RDONLY | O_NONBLOCK);

        if (0 > fd )
        {
             routingGstreamer.error(__FUNCTION__, ":Could not open file descriptor");
             retvalue = E_NOT_POSSIBLE;
        }
        else
        {
            routingGstreamer.info(__FUNCTION__, ":set file descriptor of fdsrc fd=", fd);

            /* flush buffer, so "old" data is deleted */
            ssize_t numberOfReadByte = 0;
            uint16_t bytesToRead = 60000; /* typical a full buffer contains 56640 bytes */
            uint8_t buffer[bytesToRead];
            uint8_t attempts = 0;
            do
            {
                numberOfReadByte = read(fd, &buffer, bytesToRead);
                routingGstreamer.info(__FUNCTION__, ":read ",numberOfReadByte, "buffer[0] ",  buffer[0]);
                attempts++;
            }while((numberOfReadByte > 0) && (5 >= attempts));

            if(5 == attempts)
            {
                 routingGstreamer.error(__FUNCTION__, ":Buffer not empty continue anyway");
            }

            g_object_set (G_OBJECT (bt_source), "fd", fd, NULL);
        }
        gst_object_unref(bt_source);
    }
    else
    {
        routingGstreamer.error(__FUNCTION__, ":Could not get btSource");
        retvalue = E_NOT_POSSIBLE;
    }

    return retvalue;
}

am_Error_e CAmRoutingSenderGstreamer::closeFileDescriptorSource(RoutingSenderGstConnection* connection)
{
	am_Error_e retvalue = E_NOT_POSSIBLE;
	GstElement *bt_source = nullptr;
	gint fd, ret;

	if(!connection && !connection->src->element)
		return retvalue;

	routingGstreamer.info(__FUNCTION__, ":Close file descriptor BT-Source");

	bt_source = gst_bin_get_by_name(GST_BIN(connection->src->element), "btSource");
	if (bt_source)
	{
		g_object_get (G_OBJECT (bt_source), "fd", &fd, NULL);
		ret = close(fd);
		if(ret < 0)
		{
			routingGstreamer.error(__FUNCTION__, ":failed: closing btsource file descriptor", connection->id);
		}
		else
		{
			retvalue = E_OK;
			routingGstreamer.info(__FUNCTION__, ":passed: closing btsource file descriptor", connection->id);
		}

		gst_object_unref(bt_source);
	}
	else
	{
		routingGstreamer.error(__FUNCTION__, ":No element named btSource in connection id: ", connection->id);
	}

	return retvalue;
}

/**
 * @brief This function checks if the desired GstElement has changed to the new state.
 * @param[in] *object The Gstreamer-Object/Element which has changed the state.
 * @param[in] new_state The new state of the Gstreamer-Object/Element.
 * @return void
 */
void CAmRoutingSenderGstreamer::dispatchStateChange(GstObject *object, GstState new_state)
{
    const auto iterator = m_state_callbacks.find((GstElement *) object);
    if (m_state_callbacks.end() == iterator)
    {
        /* Not the elemet we waiting for -> continue*/
        return;
    }

    if (new_state != std::get<0>(iterator->second))
    {
        routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChange -- new state ", new_state, " doesn't fit state ",
                              (std::get<0>(iterator->second)));
        return;
    }

    StateChangeFunction callbackFunction = std::get<1>(iterator->second);
    void* callbackData                   = std::get<2>(iterator->second);

    /* Callback-Function is always sendAckSetSourceState as of today 07.02.2016 */
    if(callbackFunction)
    {
        routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChange -- call callback function");
        (this->*callbackFunction)(new_state, callbackData);
    }
    m_state_callbacks.erase(iterator);
    routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChange -- finished");
}

void CAmRoutingSenderGstreamer::dispatchStateChangeEOS(GstObject *object, GstState new_state)
{
    routingGstreamer.info(" dispatchStateChangeEOS object, state", object, new_state);
    const auto iterator = m_state_callbacks.find((GstElement *) object);
    if (m_state_callbacks.end() == iterator)
    {
        /* When there is no callback check if the source of the current connection triggered the EOS */
        auto connection = m_gstController->getAncestorConnectionItemFromObject(object);

        /* check if the given object is part of the pipeline */
        if(nullptr != connection)
        {
            routingGstreamer.info(__FUNCTION__,": EOS is part of current pipe with source id: ", connection->src->source.sourceID, " with flag ", connection->src->flags);

            if(std::string::npos != connection->src->flags.find("GST_RA_FILE"))
            {
                routingGstreamer.info("CAmRoutingSenderGstreamer::EOS triggered by file source -> send sourceNotification NT_SOURCE_EOS / 1");

                am_NotificationPayload_s notificationPayload = {
                .type = NT_SOURCE_EOS,
				.value = 1
                };

                this->mpCAmRoutingReceiverShadow->hookSourceNotificationDataChange(connection->src->source.sourceID, notificationPayload);
            }
        }
        else
        {
            routingGstreamer.error("AmRoutingSenderGstreamer::dispatchStateChangeEOS: no callback and pipe is ot part of current pipe. Not expected. ");
        }
        return;
    }

    /* there is a valid callback which shall be handled */
    if (new_state != std::get<0>(iterator->second))
    {
        routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChangeEOS -- new state ", new_state, " doesn't fit state ",
                              (std::get<0>(iterator->second)));
        return;
    }

    StateChangeFunction callbackFunction = std::get<1>(iterator->second);
    void* callbackData                   = std::get<2>(iterator->second);
    std::tuple<RoutingSenderGstConnection *, am_Handle_s, GstState> *data = static_cast<std::tuple<RoutingSenderGstConnection *, am_Handle_s, GstState>*>(callbackData);
    am_Handle_s handleLocal = std::get<1>(*data);
    auto connection = std::get<0>(*data);

    if(callbackFunction)
    {
        routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChangeEOS -- call callback function");
        this->changeElementState(connection->pipeline, new_state, callbackFunction, callbackData, connection->src->element, handleLocal);
    }
    m_state_callbacks.erase(iterator);
    routingGstreamer.info("CAmRoutingSenderGstreamer::dispatchStateChangeEOS -- finished");
}



/**
 * @brief Preparation of the Gstreamer-Routing-Plugin
 *        Prepare the domain, sinks and sources. Initialization the Alsa monitoring.
 */
bool CAmRoutingSenderGstreamer::registerGamElements(const std::string &busName, am_domainID_t& domainId)
{
    routingGstreamer.info("CAmRoutingSenderGstreamer::registerGamElements");

    if (!m_gstController->registerDomain(busName, domainId) || !m_gstController->registerAllSinks() || !m_gstController->registerAllSources())
        return false;

    m_gstController->startBtAvailabilityMonitoring();
    return true;
}

/**
 * @brief Contains the shutdown sequence of the Gstreamer-Routing-Plugin
 *        Deregistration of domain, sinks and sources. Stopping the Alsa monitoring.
 */
bool CAmRoutingSenderGstreamer::deregisterGamElements()
{
    routingGstreamer.info(__FUNCTION__, ":setRoutingRundown called");

    mpspi_tclMySPINSndCtrlMonitor->vStopMonitor();

    if(!m_gstController->deregisterAllSources() || !m_gstController->deregisterAllSinks() || !m_gstController->deregisterDomain())
        return false;

    return true;
}

/**
 * @brief Contains the abort sequence of the Gstreamer-Routing-Plugin
 *        Clearing the callback map
 * @param[in] handle Handle
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncAbort(const am_Handle_s handle)
{
	routingGstreamer.info(__FUNCTION__, ":asyncAbort called");

	if (H_SETSOURCESTATE == handle.handleType)
	{
		routingGstreamer.info(__FUNCTION__, ":Clear m_state_callbacks");

		this->m_state_callbacks.clear();
	}

    return (E_OK);
}

/**
 * @brief Contains the connection sequence
 *        Connects a source (sourceID) with a sink (sinkID).
 * @param[in] handle Handle
 * @param[in] connectionID ID of the connection
 * @param[in] sourceID ID of the source
 * @param[in] sinkID ID of the sink
 * @param[in] connectionFormat Connection format
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncConnect(const am_Handle_s handle, const am_connectionID_t connectionID, const am_sourceID_t sourceID,
        const am_sinkID_t sinkID, const am_CustomAvailabilityReason_t connectionFormat)
{
    (void) connectionFormat;

	routingGstreamer.info(__FUNCTION__);

    am_Error_e retvalue = E_OK;
    auto sourceItem = m_gstController->getSourceItem(sourceID);
    auto sinkItem = m_gstController->getSinkItem(sinkID);

    RoutingSenderGstConnection *localConnection = new RoutingSenderGstConnection();
    GstState currentState, pending;

    if(nullptr == sinkItem)
    {
        routingGstreamer.error(__FUNCTION__, ": wrong sink ID ", sinkID);
        retvalue = E_NON_EXISTENT;
    }

    if(nullptr == sourceItem)
    {
        routingGstreamer.error(__FUNCTION__, ": only static source pads are supported, source ID  ", sourceID);
        retvalue = E_NON_EXISTENT;
    }

    if(E_OK == retvalue)
    {
        localConnection->id = connectionID;
        localConnection->src = sourceItem;
    	localConnection->sink = sinkItem;

    	RoutingSenderGstConnection* pipeOfSink = m_gstController->getConnectionItemFromSinkId(sinkID);
    	if(nullptr != pipeOfSink)
    	{
    	    routingGstreamer.info(__FUNCTION__, ": sink is already part of a pipe");
    	    localConnection->pipeline = pipeOfSink->pipeline;
    	    localConnection->bus = pipeOfSink->bus;
    	}
    	else
    	{
    	    /* sink is not part of a pipe, create new one */
    	    if(E_OK != setNewPipe(connectionID, localConnection))
            {
                routingGstreamer.error(__FUNCTION__, ": setNewPipe: failed");
                retvalue = E_NOT_POSSIBLE;
            }
    	}
    }

    if(E_OK == retvalue)
    {
    	m_gstController->addConnection(connectionID, localConnection->src, localConnection->sink, localConnection->pipeline, localConnection->bus);

		/* wait if there is an asynchronously e.g. sample rate change ongoing  */
		if (gst_element_get_state(localConnection->pipeline, &currentState, &pending, 100 * GST_MSECOND)
						== GST_STATE_CHANGE_FAILURE)
		{
			routingGstreamer.error(__FUNCTION__, ": Change element state asynchronously failed");
		}
		routingGstreamer.info(__FUNCTION__, ": currentState ", currentState, "pending ", pending);

	     if(E_OK != fillCurrentPipeWithElements(connectionID))
        {
            routingGstreamer.error(__FUNCTION__, ": fillCurrentPipeWithElements: failed");
            retvalue = E_NOT_POSSIBLE;
        }

	   delete localConnection;

	   routingGstreamer.info(__FUNCTION__, " finished");
    }
    else
    {
    	routingGstreamer.error(__FUNCTION__, " failed");
    }

    if(E_OK != retvalue)
    {
        routingGstreamer.error(__FUNCTION__, ": fillCurrentPipeWithElements failed");
    }

    mpCAmRoutingReceiverShadow->ackConnect(handle, connectionID, retvalue);
    return retvalue;
}

/**
 * @brief Contains the disconnection sequence
 * @param[in] handle Handle
 * @param[in] connectionID ID of the connection
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncDisconnect(const am_Handle_s handle, const am_connectionID_t connectionID)
{
    routingGstreamer.info(__FUNCTION__, ":asyncDisconnect called");
    auto connection = m_gstController->getConnectionItem(connectionID);

    am_Error_e retvalue = E_OK;

    retvalue = clearGhostPad(connection);
    retvalue = removeSourceFromPipe(connection);

    if(1 == m_gstController->getNumberOfConnectionsForSink(connection->sink->element))
    {
       routingGstreamer.info(__FUNCTION__, ": last connection on sink, clear all.");

       retvalue = removeSinkFromPipe(connection);

       if(E_OK != deletePipe(connectionID))
       {
           routingGstreamer.error(__FUNCTION__, ": deletePipe failed");
           retvalue = E_NOT_POSSIBLE;
       }
    }

    m_gstController->removeConnection(connectionID);
    this->mpCAmRoutingReceiverShadow->ackDisconnect(handle, connectionID, retvalue);

    debugPrintGStreamerPipeToFile(connection);

    if(E_OK != retvalue)
        routingGstreamer.error(__FUNCTION__, ": asyncDisconnect failed");

    return retvalue;
}

/**
 * @brief Responsible for ramping up/down the volume of the sink or source in a own thread.
 * @param[in] handle Handle
 * @param[in] volume new volume of the sink (Unit: MainVolume)
 * @param[in] time ramping time
 * @param[in] *volume_element Pointer to the gstreamer volume element
 * @param[in] new_volume new value for the volume_element
 * @param[in] actual_volume actual volume of the volume_element
 * @param[in] isSink True=Sink, False=Source
 * @return void
 */
void CAmRoutingSenderGstreamer::rampingNewThread(am_Handle_s handle, am_volume_t volume, am_time_t time,
        GstElement* volume_element, gdouble new_volume, gdouble actual_volume, bool isSink)
{
    gdouble delta_volume = new_volume-actual_volume;

    gint ramping_time = time;
    for(int i = 1; i <= ramping_time; i++)
    {
        new_volume = actual_volume + (gdouble) i / ramping_time * delta_volume;
        /* For testing purposes
        routingGstreamer.info(__FUNCTION__, ":actual volume: ", actual_volume, "  new_volume: ", new_volume, "  delta_volume: ", delta_volume);
        */
        g_object_set(G_OBJECT(volume_element), "volume", new_volume, NULL);
        g_usleep(1000);
    }

    gst_object_unref(volume_element);

    if (isSink)
    {
        this->mpCAmRoutingReceiverShadow->ackSetSinkVolumeChange(handle, volume, E_OK);
    }
    else
    {
        this->mpCAmRoutingReceiverShadow->ackSetSourceVolumeChange(handle, volume, E_OK);
    }
}

/**
 * @brief Contains the disconnection sequence
 * @param[in] handle Handle
 * @param[in] sinkID ID of the sink
 * @param[in] volume New volume level
 * @param[in] ramp   Ramp type
 * @param[in] time   Time for ramping up/down
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncSetSinkVolume(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume,
        const am_CustomRampType_t ramp, const am_time_t time)
{
    routingGstreamer.info(__FUNCTION__, ":asyncSetSinkVolume called");

    am_Error_e retvalue = E_OK;
    auto sinkItem = m_gstController->getSinkItem(sinkID);
    GstElement *volume_element = NULL;
    gfloat new_volume = 0;
    am_volume_t mappedVolume = volume + 1000;

    /* Check and map the volume */
    if(mappedVolume > 1000)
    {
        routingGstreamer.error(__FUNCTION__, ":Volume out of range [0;1000]. Volume: ", mappedVolume, "set to vol=1000/1000");
        new_volume = 1;
    }
    else if ((mappedVolume < 0) )
    {
        routingGstreamer.error(__FUNCTION__, ":Volume out of range [0;1000]. Volume: ", mappedVolume, "set to vol=0/1000");
        new_volume = 0;
    }
    else
    {
        new_volume = gfloat(mappedVolume)/gfloat(1000);
        routingGstreamer.info("volume: ", mappedVolume, " new_volume: ", new_volume);
    }

    if(!sinkItem)
    {
        routingGstreamer.error(__FUNCTION__, ":no sink with ID ", sinkID);
        retvalue = E_NON_EXISTENT;
    }
    else
    {
        volume_element = gst_bin_get_by_name(GST_BIN(sinkItem->element), "volume");
        if(!volume_element)
        {
            routingGstreamer.error(__FUNCTION__, ":Could not get sink volume element");
            retvalue = E_NON_EXISTENT;
        }
    }
    am_CustomRampType_t ramp1 =  ramp; /* RAMP_GENIVI_DIRECT; */

    if(E_OK == retvalue)
    {
        gdouble actual_volume;
        g_object_get(G_OBJECT(volume_element), "volume", &actual_volume, NULL);

        /* Maybe better in a own thread */
        switch(ramp1)
        {
            case RAMP_GENIVI_DIRECT:
            {
                routingGstreamer.info(__FUNCTION__, ":RAMP_GENIVI_DIRECT");
                routingGstreamer.info(__FUNCTION__, ":New Value: ", new_volume * 100, "%", "actual volume:", actual_volume*100, "%");

                if(this->threadChangeVolume)
                {
                    this->threadChangeVolume->join();
                    delete this->threadChangeVolume;
                    this->threadChangeVolume=NULL;
                }
                this->threadChangeVolume = new std::thread(&CAmRoutingSenderGstreamer::rampingNewThread, this,
                		handle, volume, 16, volume_element, new_volume, actual_volume, true);
                break;
            }
            case RAMP_GENIVI_LINEAR:
            {
                routingGstreamer.info(__FUNCTION__, ":RAMP_GENIVI_LINEAR");
                if(this->threadChangeVolume)
                {
                    this->threadChangeVolume->join();
                    delete this->threadChangeVolume;
                    this->threadChangeVolume=NULL;
                }
                this->threadChangeVolume = new std::thread(&CAmRoutingSenderGstreamer::rampingNewThread, this,
                		handle, volume, time, volume_element, new_volume, actual_volume, true);
                break;
            }
            default:
            {
                routingGstreamer.error(__FUNCTION__, ":Only RAMP_GENIVI_DIRECT or RAMP_GENIVI_LINEAR are supported. Ramp: ", ramp);

                retvalue = E_NOT_POSSIBLE;
                gst_object_unref(volume_element);
                break;
            }
        }
    }

//    am_NotificationPayload_s notificationPayload = {
//    .type = NT_SINKVOLUME,
//    .value = volume
//    };
//    this->mpCAmRoutingReceiverShadow->hookSinkNotificationDataChange(sinkID, notificationPayload);

    return retvalue;
}


/**
 * @brief Contains the disconnection sequence
 * @param[in] handle Handle
 * @param[in] sourceID ID of the source
 * @param[in] volume New volume level
 * @param[in] ramp   Ramp type
 * @param[in] time   Time for ramping up/down
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncSetSourceVolume(const am_Handle_s handle, const am_sourceID_t sourceID, const am_volume_t volume,
        const am_CustomRampType_t ramp, const am_time_t time)
{
    routingGstreamer.info(__FUNCTION__, ":asyncSetSourceVolume called");

    am_Error_e retvalue = E_OK;
    auto sourceItem = m_gstController->getSourceItem(sourceID);
    GstElement *volume_element;
    gfloat new_volume = 0;
    am_volume_t mappedVolume = volume + 1000;

    /* Check and map the volume */
    if(mappedVolume > 1000)
    {
        routingGstreamer.error(__FUNCTION__, ":Volume out of range [0;1000]. Volume: ", mappedVolume, "set to vol=1000/1000");
        new_volume = 1;
    }
    else if ((mappedVolume < 0) )
    {
        routingGstreamer.error(__FUNCTION__, ":Volume out of range [0;1000]. Volume: ", mappedVolume, "set to vol=0/1000");
        new_volume = 0;
    }
    else
    {
        new_volume = gfloat(mappedVolume)/gfloat(1000);
        routingGstreamer.info("volume: ", mappedVolume, " new_volume: ", new_volume);
    }

    if(!sourceItem)
    {
        routingGstreamer.error(__FUNCTION__, ":no source with ID ", sourceID);
        retvalue = E_NON_EXISTENT;
    }
    else
    {
        volume_element = gst_bin_get_by_name(GST_BIN(sourceItem->element), "volume");
        if(!volume_element)
        {
            routingGstreamer.error(__FUNCTION__, ":Could not get source volume element");
            retvalue = E_NON_EXISTENT;
        }
    }

    am_CustomRampType_t ramp1 = ramp; /* RAMP_GENIVI_DIRECT; */
    if(E_OK == retvalue)
    {
        gdouble actual_volume;
        g_object_get(G_OBJECT(volume_element), "volume", &actual_volume, NULL);

        /* Maybe better in a own thread */
        switch(ramp1)
        {
            case RAMP_GENIVI_DIRECT:
            {
                routingGstreamer.info(__FUNCTION__, ":RAMP_GENIVI_DIRECT");
                routingGstreamer.info(__FUNCTION__, ":New Value: ", new_volume * 100, "%", "actual volume:", actual_volume*100, "%");

                if(this->threadChangeSourceVolume)
                {
                    this->threadChangeSourceVolume->join();
                    delete this->threadChangeSourceVolume;
                    this->threadChangeSourceVolume=NULL;
                }
                this->threadChangeSourceVolume = new std::thread(&CAmRoutingSenderGstreamer::rampingNewThread, this, handle, volume, 16,
                        volume_element, new_volume, actual_volume, false);
                break;
            }
            case RAMP_GENIVI_LINEAR:
            {
                routingGstreamer.info(__FUNCTION__, ":RAMP_GENIVI_LINEAR");
                if(this->threadChangeSourceVolume)
                {
                    this->threadChangeSourceVolume->join();
                    delete this->threadChangeSourceVolume;
                    this->threadChangeSourceVolume=NULL;
                }
                this->threadChangeSourceVolume = new std::thread(&CAmRoutingSenderGstreamer::rampingNewThread, this, handle, volume, time,
                        volume_element, new_volume, actual_volume, false);
                break;
            }
            default:
            {
                routingGstreamer.error(__FUNCTION__, ":Only RAMP_GENIVI_DIRECT or RAMP_GENIVI_LINEAR are supported. Ramp: ", ramp);

                retvalue = E_NOT_POSSIBLE;
                gst_object_unref(volume_element);
                break;
            }
        }
    }

    /* check if the ramp is not set successfully. In this case directly respond */
    if(E_OK != retvalue)
    {
    	routingGstreamer.error(__FUNCTION__, ":asyncSetSourceVolume failed");

    	this->mpCAmRoutingReceiverShadow->ackSetSourceVolumeChange(handle, volume, retvalue);
    }

//    am_NotificationPayload_s notificationPayload = {
//    .type = NT_SOURCEVOLUME,
//    .value = volume
//    };
//    this->mpCAmRoutingReceiverShadow->hookSourceNotificationDataChange(sourceID, notificationPayload);

    return retvalue;
}

GstStateChangeReturn CAmRoutingSenderGstreamer::syncChangeElementState(GstElement* element, GstState stateToSet)
{
	GstStateChangeReturn eStateChangeReturn = GST_STATE_CHANGE_SUCCESS;
	GstState currentState, pending;

	routingGstreamer.info(__FUNCTION__, ":syncChangeElementState element", element," stateToSet: ", stateToSet);

	if(nullptr != element)
	{
		eStateChangeReturn = gst_element_set_state(element, stateToSet);

		switch(eStateChangeReturn)
		{
			case GST_STATE_CHANGE_FAILURE:
			{
				routingGstreamer.error(__FUNCTION__, ":Change element state failed");
			}break;
			case GST_STATE_CHANGE_SUCCESS:
			{
				routingGstreamer.info(__FUNCTION__, ":Change element state success");
			}break;
			case GST_STATE_CHANGE_ASYNC:
			{
				routingGstreamer.info(__FUNCTION__, ":Change element state asynchronously");
				/* Wait until change is finished */
				if (gst_element_get_state(element, &currentState,
						&pending, 100 * GST_MSECOND) == GST_STATE_CHANGE_FAILURE)
				{
					routingGstreamer.error(__FUNCTION__, ":Change element state asynchronously failed");
				}
				routingGstreamer.info(__FUNCTION__, ":Change element state asynchronously finished cur/pen", currentState, pending);
			}break;
			case GST_STATE_CHANGE_NO_PREROLL:
			{
				routingGstreamer.warn(__FUNCTION__, ":Change element state No Preroll");
			}break;
			default:
			{
				routingGstreamer.error(__FUNCTION__, ":Change element state not handled");
			}
		}
	}
	else
	{
		routingGstreamer.error(__FUNCTION__, ":syncCES: pipe is null");
		eStateChangeReturn = GST_STATE_CHANGE_FAILURE;
	}

	return eStateChangeReturn;
}


/**
 * @brief Contains the reconfiguration sequence after a changement of the sample rate
 * @param[in] u32SampleRate the current sample rate
 * @param[in] bEndPointEnabled audio stream active or passive
 * @return void
 */
void CAmRoutingSenderGstreamer::reconfigurePipeline(am_sourceID_t sourceID, uint32_t u32SampleRate, bool bEndPointEnabled)
{
	routingGstreamer.info(__FUNCTION__, ":reconfigurePipeline called. Old SR=", this->u32SampleRateOld,  "New SR=",
			u32SampleRate, " RetState=", this->bReturnState," bEndPointEnabled=", bEndPointEnabled, "sourceID:",sourceID);

	GstState state, pending;
	auto connection = m_gstController->getConnectionItemFromSourceId(sourceID);

	routingGstreamer.info("GStreamer before lock in reConPipe ");
	sampleRateChangeLock.lock();
	routingGstreamer.info("GStreamer after lock in reConPipe ");

	if (nullptr == connection)
	{
		routingGstreamer.error("GStreamer reConPipe: No connection available: exit.");
		return;
	}

	gst_element_get_state(connection->pipeline, &state, &pending, 0);
	routingGstreamer.info(__FUNCTION__, ":Current state is: ", state, ", pending: ", pending);

	/* Change state to NULL */
	if ( !bEndPointEnabled && (this->u32SampleRateOld != u32SampleRate))
	{
		routingGstreamer.info(__FUNCTION__, ":Change element state to GST_STATE_NULL ");

		gst_element_set_locked_state(connection->src->element, false);

		if(GST_STATE_CHANGE_FAILURE == syncChangeElementState (connection->pipeline, GST_STATE_NULL))
		{
			routingGstreamer.info(__FUNCTION__, ":Change element state to GST_STATE_NULL FAILED");
		}
		else
		{
			this->u32SampleRateOld = u32SampleRate;
		}

		gst_element_set_locked_state(connection->src->element, true);

		this->bReturnState = true;

	} /* if ( !bEndPointEnabled && (this->u32SampleRateOld != u32SampleRate)) */

	/* Change state back to originalState */
	if (bEndPointEnabled )
	{
		GstState stateToSet = connection->expectedState;

		routingGstreamer.info("GStreamer reconfigure Pipe -- Change element state back to: ", stateToSet, ", 1,2,3,4 = Null, Ready, Paused, Playing");

		gst_element_set_locked_state(connection->src->element, false);

		if(GST_STATE_NULL != state)
		{
			/* State NULL transition necessary to get the pipe to the playing state in case of SRC req */
			routingGstreamer.info("GStreamer pipe not NULL, perform state transition ");

			(void)syncChangeElementState (connection->pipeline,GST_STATE_NULL);
		}

		if(GST_STATE_CHANGE_FAILURE == syncChangeElementState (connection->pipeline, stateToSet))
		{
			/* GStreamer pipe reestablish failed, go back to NULL */
			/* Do not go back to Null to not loose the final state */
			routingGstreamer.error("GStreamer reConPipe: syncChangeElementState failed ");
		}

		this->u32SampleRateOld = u32SampleRate;
	} /* if (bEndPointEnabled && (this->u32SampleRateOld == u32SampleRate) && this->bReturnState) */

	routingGstreamer.info("GStreamer before unlock in reConPipe ");
	sampleRateChangeLock.unlock();
	routingGstreamer.info("GStreamer after unlock in reConPipe ");
}

/**
 * @brief Set state of Gstreamer-object/element
 * @param[in] *element Gstreamer element/object
 * @param[in] state State whicht should be set
 * @param[in] handle Handle
 * @return void
 */
void CAmRoutingSenderGstreamer::setElementState(GstElement *element, GstElement *sourceBin, GstState state, am_Handle_s handle)
{
	am_Error_e error = E_OK;
	routingGstreamer.info(__FUNCTION__, ":setElementState called, element : ", element, ", state: ", state);

	routingGstreamer.info("GStreamer before lock in sES ");
	sampleRateChangeLock.lock();
	routingGstreamer.info("GStreamer after lock in sES ");

	if(nullptr == element)
	{
		routingGstreamer.error(__FUNCTION__, ":setElementState failed. element ", element);
		error = E_NOT_POSSIBLE;
	}
	else
	{
		GstStateChangeReturn setElemetErrorCode = gst_element_set_state(element, state);
		routingGstreamer.info("CAmRoutingSenderGstreamer::setElementState -- return code ", setElemetErrorCode);

		switch (setElemetErrorCode)
		{
		case GST_STATE_CHANGE_SUCCESS:
        case GST_STATE_CHANGE_ASYNC:
            break;
        case GST_STATE_CHANGE_FAILURE:
        case GST_STATE_CHANGE_NO_PREROLL:
        default:
            routingGstreamer.info("CAmRoutingSenderGstreamer::setElementState -- reporting back error");
            error = E_NOT_POSSIBLE;
		}
	}

	/* in case there is an error check if there is an handled and a sourceBin to send an ack and clean up */
	if((E_OK != error) && (0 != handle.handle) && (nullptr != sourceBin))
	{
		/* directly send ack in error case*/
		routingGstreamer.info(__FUNCTION__, ":Remove callback ");
		auto itstate_callback = this->m_state_callbacks.find(sourceBin);

		if(this->m_state_callbacks.end() != itstate_callback)
		{
			this->m_state_callbacks.erase(itstate_callback);
			routingGstreamer.info(__FUNCTION__, ":Callback removed ");
		}

		this->mpCAmRoutingReceiverShadow->ackSetSourceState(handle, error);
	}

	routingGstreamer.info("GStreamer before unlock in sES ");
	sampleRateChangeLock.unlock();
	routingGstreamer.info("GStreamer after unlock in sES ");
}

/**
 * @brief Set state of Gstreamer-object/element
 * @param[in] *element Gstreamer element/object
 * @param[in] state State which should be set
 * @param[in] callback callback Function
 * @param[in] *user_data User data
 * @param[in] *sourceBin pointer to the source bin
 * @param[in] handle Handle
 * @return void
 */
void CAmRoutingSenderGstreamer::changeElementState(GstElement *element, GstState state,
		StateChangeFunction callback, void *user_data, GstElement *sourceBin, am_Handle_s handle)
{
    routingGstreamer.info(__FUNCTION__, ":changeElementState called");
    routingGstreamer.info(__FUNCTION__, ":Element: ", element, ", state: ", state, ", sourceBin: ", sourceBin);

    std::tuple<GstState, StateChangeFunction, void *> state_callback;

    state_callback = std::make_tuple(state, callback, user_data);

    this->m_state_callbacks[sourceBin] = state_callback;

	// FIXME: using the serializer is somehow hackish
    this->mpCAmRoutingReceiverShadow->getSerializer()->asyncCall(this, &CAmRoutingSenderGstreamer::setElementState,
                                                                 element, sourceBin, state, handle);

}

/**
 * @brief Set Calls sendAckSetSourceState
 * @param[in] state New state of source
 * @param[in] *data Contains Handle
 * @return void
 */
void CAmRoutingSenderGstreamer::sendAckSetSourceState(GstState state, std::tuple<RoutingSenderGstConnection *, am_Handle_s, GstState> *data)
{
    routingGstreamer.info(__FUNCTION__, ": called");

    RoutingSenderGstConnection *connection = std::get< 0 >(*data);
    am_Handle_s handleLokal = std::get< 1 >(*data);
    GstState finalStateLokal = std::get< 2 >(*data);

    switch(state)
    {
        case GST_STATE_READY:
        	/* set the source state to unkown as intermediate result till reaching playing */
        	connection->src->source.sourceState = SS_UNKNNOWN;
            this->threadChangeBTSourceState = new std::thread(
                &CAmRoutingSenderGstreamer::workerThreadChangeState, this, connection->pipeline, finalStateLokal, handleLokal);
            break;
        case GST_STATE_PLAYING:
        	connection->src->source.sourceState = SS_ON;
            routingGstreamer.info(__FUNCTION__, ":SourceState = SS_ON");
            break;
        case GST_STATE_PAUSED:
        	connection->src->source.sourceState = SS_PAUSED;
            routingGstreamer.info(__FUNCTION__, ":SourceState = SS_PAUSED");
            break;
        case GST_STATE_NULL:
        	connection->src->source.sourceState = SS_OFF;
            routingGstreamer.info(__FUNCTION__, ":SourceState = SS_OFF");
            break;
        default:
            routingGstreamer.error(__FUNCTION__, ":state is not SS_ON, SS_OFF or SS_PAUSED");
            break;
    }

    this->mpCAmRoutingReceiverShadow->ackSetSourceState(handleLokal, E_OK);
    delete data;
}

/**
 * @brief Worker thread which change the state to the final state playing
 * @param[in] *element Pointer to Gstreamer element/pipeline
 * @param[in] state Final state of BT-source
 * @return void
 */
void CAmRoutingSenderGstreamer::workerThreadChangeState(GstElement *element, GstState state, am_Handle_s handle)
{
	routingGstreamer.info(__FUNCTION__, "workerThreadChangeState called for ", element, ", state ", state, " and handle ", handle);
	GstState currentState, pendingState;

	if(nullptr != element)
	{
		gst_element_get_state(element, &currentState, &pendingState, 100 * GST_MSECOND);

		routingGstreamer.info(__FUNCTION__, "Current state: ", currentState, ", pending state: ", pendingState);

		if(GST_STATE_PLAYING == state)
		{
			/* Common part of final change to playing state */
			routingGstreamer.info(__FUNCTION__, "Continue changing state to PLAYING state: ", state);
			gst_element_set_locked_state(element, false);

			this->setElementState(element, NULL, state, handle);
			gst_element_get_state(element, &currentState, &pendingState, 100 * GST_MSECOND);

			routingGstreamer.info(__FUNCTION__, "State change finished to PLAYING state for element: ", element);

			/* sync state to pipeline */
			auto connectionSrc = m_gstController->getConnectionItemFromSrcElement(element);

			if(nullptr != connectionSrc)
			{
	            this->setElementState(connectionSrc->pipeline, NULL, state, handle);
	            gst_element_get_state(connectionSrc->pipeline, &currentState, &pendingState, 100 * GST_MSECOND);
	            routingGstreamer.info(__FUNCTION__, "State change finished to PLAYING state for pipe of element: ", connectionSrc->pipeline);

	            debugPrintGStreamerPipeToFile(connectionSrc);
			}
			else
			    routingGstreamer.info(__FUNCTION__, "not the source element, so pipe is ok");
		}
		else
			routingGstreamer.error(__FUNCTION__, ":worker-thread unhandled state: ", state);
	}
	else
		routingGstreamer.error(__FUNCTION__, ":worker-thread: found connection is null ");

	routingGstreamer.info(__FUNCTION__, ":leave workerThreadChangeState");
}

/**
 * @brief Contains the Set-Source-State sequence
 * @param[in] handle Handle
 * @param[in] sourceID ID of the source
 * @param[in] state State which should be set
 * @return Error code
 */
am_Error_e CAmRoutingSenderGstreamer::asyncSetSourceState(const am_Handle_s handle, const am_sourceID_t sourceID, const am_SourceState_e state)
{
    routingGstreamer.info(__FUNCTION__, ":asyncSetSourceState called");

    am_Error_e retvalue = E_OK;
	GstState currentState, pending;

    auto connection = m_gstController->getConnectionItemFromSourceId(sourceID);
    routingGstreamer.info(__FUNCTION__, ":Connection with id found: ", connection->id);
    debugPrintGStreamerPipeToFile(connection);

	if(nullptr != connection)
	{
		if(sourceID != connection->src->source.sourceID)
		{
			routingGstreamer.error(__FUNCTION__, ":Source ID is not as expected.");
			retvalue = E_NON_EXISTENT;
		}
		else
		{
		    /* wait if there is an asynchronously e.g. sample rate change ongoing  */
			if (gst_element_get_state(connection->src->element , &currentState, &pending, 100 * GST_MSECOND)
					== GST_STATE_CHANGE_FAILURE)
			{
				routingGstreamer.error(
						__FUNCTION__, ":ASSS: Change element state asynchronously failed");
			}
			routingGstreamer.info(__FUNCTION__, ":asyncSetSourceState currentState", currentState, "pending", pending);

	        if(this->m_state_callbacks.end() != this->m_state_callbacks.find(connection->src->element))
	        {
	            routingGstreamer.error(__FUNCTION__, ":source ID ", sourceID, "is not done changing state");
	            retvalue = E_NOT_POSSIBLE;
	        }
		}
	}
	else
	{
		routingGstreamer.error(__FUNCTION__, ":No active connection found.");
		retvalue = E_NON_EXISTENT;
	}

    if(E_OK == retvalue)
    {
    	if (this->threadChangeBTSourceState)
    	{
    		/* End current actions */
	        routingGstreamer.info(__FUNCTION__, ":stop Thread ChangeBTSourceState");
	        this->bStopThreadChangeBTSourceState = true;
	        this->threadChangeBTSourceState->join();
	        delete this->threadChangeBTSourceState;
	        this->threadChangeBTSourceState=NULL;
	        this->bStopThreadChangeBTSourceState = false;
	        routingGstreamer.info(__FUNCTION__, ":stopped Thread ChangeBTSourceState");

	        /* ensure that the currentState is set as given for all elements */
    		syncChangeElementState(connection->src->element, currentState);
    		if (gst_element_get_state(connection->pipeline, &currentState, &pending, 100 * GST_MSECOND)
    					== GST_STATE_CHANGE_FAILURE)
			{
				routingGstreamer.error(__FUNCTION__, ":ASSS: Change element state asynchronously failed");
			}
    		else
    		{
    			routingGstreamer.info(__FUNCTION__, ":after syncChangeElementState currentState", currentState, "pending", pending);
    		}
	    }

        switch(state)
        {
            case SS_ON:
            {
            	if(!m_gstController->setConnectionState(connection->id, GST_STATE_PLAYING))
            	{
            		routingGstreamer.error(__FUNCTION__, ":Could not set connection state");
            	}

            	/* Check if both src and sink are part of the pipeline */
                if(!gst_object_has_ancestor(GST_OBJECT(connection->src->element), GST_OBJECT(connection->pipeline)) &&
                		gst_object_has_ancestor(GST_OBJECT(connection->sink->element), GST_OBJECT(connection->pipeline)))
                {
                    routingGstreamer.error(__FUNCTION__, "GStreamer -- can't set state ON on source ID ", sourceID);
                    retvalue = E_NOT_POSSIBLE;
                }
                else
                {
                    /* We make sure the pipeline starts with the first source, but we want the
                     * acknowledgement to happen as soon as the source goes to PLAYING or READY*/
                    if(std::string::npos != connection->src->flags.find("GST_RA_LIVE_FD"))
                    {
                    	retvalue = openFileDescriptorSource(connection);
                    }
                    else if(std::string::npos != connection->src->flags.find("GST_RA_FILE"))
                    {
                    	routingGstreamer.info(__FUNCTION__, ": Set pad event function.");
						auto eosCallbackData = new std::tuple<CAmRoutingSenderGstreamer *,  am_sourceID_t >(this, sourceID);

                    	if(!gst_pad_add_event_probe(connection->sinkPad,GCallback(callback_pad_event_function), eosCallbackData))
							routingGstreamer.error(__FUNCTION__, ": gst_pad_add_event_probe failed");
                    }
                    else
                    {
                        /* no special handling for other sources defined */
                    }

                    if(E_OK == retvalue)
                    {
                        auto cb_data = new std::tuple<RoutingSenderGstConnection *, am_Handle_s, GstState>(connection, handle, GST_STATE_PLAYING);

                        this->SetSourceStateSource = sourceID;
                        gst_element_set_locked_state(connection->src->element, false);

                        GstState currentSrcState, pendingSrcState, currentSnkState, pendingSnkState, currentPipeState, pendingPipeState;

                        gst_element_get_state(connection->src->element,&currentSrcState,&pendingSrcState,100 * GST_MSECOND);
                        routingGstreamer.info(__FUNCTION__, " Source: currentState", currentSrcState, "pendingState", pendingSrcState);

                        gst_element_get_state(connection->sink->element,&currentSnkState,&pendingSnkState,100 * GST_MSECOND);
                        routingGstreamer.info(__FUNCTION__, " Sink: currentState", currentSnkState, "pendingState", pendingSnkState);

                        gst_element_get_state(connection->pipeline,&currentPipeState,&pendingPipeState,100 * GST_MSECOND);
                        routingGstreamer.info(__FUNCTION__, " Pipe: currentState", currentPipeState, "pendingState", pendingPipeState);

                        if(!isSrcSinkInConnectionLinked(connection))
                        {
                        	linkSrcSinkInConnection(connection);

                        }

                        switch(currentSnkState)
						{
							case GST_STATE_NULL:
								/* Change the state of the src element to GST_STATE_PLAYING */
								if(GST_STATE_READY == currentSrcState)
								{
									routingGstreamer.info(__FUNCTION__, ":already in ready state. Async change to playing.");
									this->threadChangeBTSourceState = new std::thread(
										&CAmRoutingSenderGstreamer::workerThreadChangeState, this, connection->src->element, GST_STATE_PLAYING, handle);
									this->mpCAmRoutingReceiverShadow->ackSetSourceState(handle, E_OK);
									delete cb_data;
								}
								else
								{
									this->changeElementState(connection->pipeline, GST_STATE_READY,
										(StateChangeFunction) &CAmRoutingSenderGstreamer::sendAckSetSourceState,
										cb_data, connection->src->element, handle);
								}
								break;

							case GST_STATE_READY:
							case GST_STATE_PAUSED:
							case GST_STATE_PLAYING:
								routingGstreamer.info(__FUNCTION__, ":sink already in use. Just sync source element with parent");

								gboolean statechange;
								statechange = gst_element_sync_state_with_parent(connection->src->element);

								if(statechange == GST_STATE_CHANGE_SUCCESS)
								{
									/* Set source state to SS_ON for sending acknowledgement to AM */
									connection->src->source.sourceState = SS_ON;
									routingGstreamer.info(__FUNCTION__, " SRC and SINK are synced to GST_STATE_PLAYING");
								}
								else
								{
									retvalue = E_NOT_POSSIBLE;
									routingGstreamer.error(__FUNCTION__, " failed  to sync SRC and SINK to GST_STATE_PLAYING");
								}
								break;

							default:
								break;
						}
                    }
                }
            }break;

            case SS_PAUSED:
            case SS_OFF:
            {
            	if(state == SS_PAUSED)
            	{
					/* setting expected state */
					if(!m_gstController->setConnectionState(connection->id, GST_STATE_PAUSED))
					{
						routingGstreamer.error(__FUNCTION__, ":Could not set expected GST_STATE_PAUSED state");
					}
            	}
            	else if(state == SS_OFF)
            	{
            		/* setting expected state */
					if(!m_gstController->setConnectionState(connection->id, GST_STATE_NULL))
					{
						routingGstreamer.error(__FUNCTION__, ":Could not set expected GST_STATE_NULL state");
					}
            	}

            	/* check if source bin is part of the pipeline */
            	if(gst_object_has_ancestor(GST_OBJECT(connection->src->element), GST_OBJECT(connection->pipeline)))
				{
            		gboolean eos_handled = false;
            		GstStateChangeReturn stateChangeRet;
            		GstState currentSrcState, pendingSrcState;

            		/* find current source state */
            		gst_element_get_state(connection->src->element,&currentSrcState,&pendingSrcState,100 * GST_MSECOND);
					routingGstreamer.info(__FUNCTION__, ": source, currentState", currentSrcState, " pendingState", pendingSrcState);

					/* separate handling for individual source state */
            		if(GST_STATE_NULL == currentSrcState)
					{
						/* already reached expected state, acknowledge and remove old state for sample rate change */
						routingGstreamer.info(__FUNCTION__, ":current state is already GST_STATE_NULL ");
						connection->src->source.sourceState = state;
						break;
					}
            		else if((GST_STATE_READY == currentSrcState) || (GST_STATE_PAUSED == currentSrcState))
            		{
            			/* no special handling for other states GST_STATE_READY || GST_STATE_PAUSED */
            		    eos_handled = true;
            		}
            		else if(GST_STATE_PLAYING == currentSrcState)
            		{
            			/* common handling, sending EOS event */
            			if(1 == m_gstController->getNumberOfConnectionsForSink(connection->sink->element))
						{
							routingGstreamer.info(__FUNCTION__, ": last connection, send EOS to entire pipeline");
							eos_handled = gst_element_send_event(connection->pipeline, gst_event_new_eos ());
						}
						else
						{
							routingGstreamer.info(__FUNCTION__, ": send EOS only to specific source.");
							eos_handled = gst_pad_send_event(connection->ghostSinkPad, gst_event_new_eos ());
						}
            		}

					/* unlink the source and sink bin from pipeline */
					routingGstreamer.info(__FUNCTION__, ": finalize with state transition + unlink.");
					if(eos_handled == true)
					{
						if(1 == m_gstController->getNumberOfConnectionsForSink(connection->sink->element))
						{
							/* set the state of entire pipeline element to NULL */
							routingGstreamer.info(__FUNCTION__, ": last connection. set state of entire pipeline to  null.");
							stateChangeRet = syncChangeElementState(connection->pipeline, GST_STATE_NULL);
						}
						else
						{
							/* set the state of only source element to NULL */
							routingGstreamer.info(__FUNCTION__, ": set source state to null.");
							if(!gst_element_set_locked_state(connection->src->element, true))
							{
								routingGstreamer.error(__FUNCTION__, ": lock source failed.");
							}
							stateChangeRet = syncChangeElementState(connection->src->element, GST_STATE_NULL);
						}

						/* source type specific handling */
						if (std::string::npos != connection->src->flags.find("GST_RA_LIVE_FD"))
						{
							retvalue = closeFileDescriptorSource(connection);
						}
						else
						{
							/* no special handling for other sources defined */
						}

						/* unlink source and sink only on GST_STATE_CHANGE_SUCCESS */
						/* TODO: code for other states are also required */
						if(stateChangeRet == GST_STATE_CHANGE_SUCCESS)
						{
							if(isSrcSinkInConnectionLinked(connection))
							{
								/* unlink the elements */
								if(!unLinkSrcSinkInConnection(connection))
								{
									routingGstreamer.error(__FUNCTION__, ": could not unlink elments.");
									retvalue = E_NOT_POSSIBLE;
								}
							}
							else
							{
								routingGstreamer.info(__FUNCTION__, ": elements are already unlinked.");
							}
						}
					}
					else
					{
						retvalue = E_NOT_POSSIBLE;
						routingGstreamer.info(__FUNCTION__, ": EOS handling failed.");
					}
				}
            	else
            	{
            		routingGstreamer.error(__FUNCTION__, ": can't set state OFF on sink ID ", sourceID,
            									", as it is not in the pipeline and unlinked");
					retvalue = E_NOT_POSSIBLE;
            	}

            	/* update the source state */
				if(retvalue == E_OK)
				{
					connection->src->source.sourceState = state;
					routingGstreamer.info(__FUNCTION__, ": set state SS_OFF on source ID ", sourceID);
				}
				else
				{
					connection->src->source.sourceState = SS_UNKNNOWN;
					routingGstreamer.error(__FUNCTION__, ": set state SS_UNKNNOWN on source ID ", sourceID);
				}

            }
            break;

            default:
            {
                retvalue = E_NON_EXISTENT;
            }
            break;
        }

        /* if sourceBin already in desired state remove callback and send ack */
        if(connection->src->source.sourceState == state)
        {
    		routingGstreamer.warn(__FUNCTION__, ":sourceID", sourceID, " already in desired state: ", state);
			routingGstreamer.info(__FUNCTION__, ":Remove callback ");
			auto itstate_callback = this->m_state_callbacks.find(connection->src->element);
            if(this->m_state_callbacks.end() != itstate_callback)
        	{
        	    this->m_state_callbacks.erase(itstate_callback);
        	    routingGstreamer.info(__FUNCTION__, ":Callback removed ");
        	}
        	this->mpCAmRoutingReceiverShadow->ackSetSourceState(handle, E_OK);
        }
    }

	routingGstreamer.info("GStreamer before unlock in aSS ");
	sampleRateChangeLock.unlock();
	routingGstreamer.info("GStreamer after unlock in aSS ");

    if(E_OK != retvalue)
    {
    	routingGstreamer.error(__FUNCTION__, ":asyncSetSourceState failed ");

    	if((E_NON_EXISTENT == retvalue) && (SS_OFF == state))
    	{
    		/* AM wants to switch it off but it is already of -> ack */
    		this->mpCAmRoutingReceiverShadow->ackSetSourceState(handle, E_OK);
    		retvalue = E_OK;
    	}
    	else
    	{
    		this->mpCAmRoutingReceiverShadow->ackSetSourceState(handle, retvalue);
    	}
    }
    debugPrintGStreamerPipeToFile(connection);
    return retvalue;
}

}
